/* $Id: rpmarm.c,v 1.6 1999/03/12 23:20:53 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Rpm arm test program */

/* This test program can be used with one input module, or with two
   input modules.  With two modules, both single and multi-module rpm
   arming can be tested, and since multi-module rpm arming is the most
   fragile it is best to use two modules.

   Both modules should have tach boards.  The tach signals should be:

   Tach0, Tach3:	100 Hz -> 500 Hz, 30 sec, square wave, 1 volt peak
   Tach1, Tach2:	 50 Hz -> 250 Hz, 30 sec, square wave, 1 volt peak

   The signal generator should be in continuous sweep mode.

   This setup can be done with a single 3326.  Without one of those, I
   think you're stuck.

   The setup for this test program is identical to the setup for the
   fourtach test program, and both are called from the rpmarm-suite QA
   test suite. */

#include <math.h>	/* For floor */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"
#include "err1432.h"

/* Note that we allow a small margin between the low end of the sweep
   and RPM_LOW, so that the RPM is not tainted by the glitch as the
   sweep goes from the end of the previous sweep to the start of the
   new sweep.  Similarly, RPM_HIGH is slightly below the actual top
   end of the sweep.

   Also, RPM_PRE_ARM must be enough below RPM_LOW that the E1432
   firmware has time to go from PRE_ARM to the measurement loop.  For
   the sweep we are doing, 200 RPM = 3.3 Hz = 0.25 sec seems to be
   enough. */
#define	BLOCKSIZE	1024
#define	RPM_HIGH	(SPAN * 60 / 10 * 0.95)
#define	RPM_LOW		(SPAN * 60 / 10 / 5 * 1.05)
#define	RPM_PRE_ARM	(RPM_LOW - 200)
#define	SPAN		5000

#define	NMOD_MAX	4
#define	NTACH_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_INPUT_CHANS + NTACH_MAX)

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: rpmarm.c,v 1.6 1999/03/12 23:20:53 ericb Exp $";
static const char *progname;

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list,
     int tach1, int tach2, int tach3, int tach4,
     int *tach_used)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    SHORTSIZ16 *tchan_list;
    int     i, nc, nt;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    nt = 0;
    for (i = 0; i < nmod; i++)
    {
	nc += hwconfig[i].input_chans;
	nt += hwconfig[i].tach_chans;
    }
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    if ((tach4 && nt < 4) || (tach3 && nt < 3) ||
	(tach2 && nt < 2) || (tach1 && nt < 1))
    {
	(void) fprintf(stderr, "%s: Requested more tach channels than exist\n",
		       progname);
	return -1;
    }

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    tchan_list = chan_list + nc;
    nt = 0;
    (void) memset(tach_used, 0, sizeof *tach_used * NTACH_MAX);
    if (tach1)
    {
	tchan_list[nt++] = E1432_TACH_CHAN(1);
	tach_used[0] = 1;
    }
    if (tach2)
    {
	tchan_list[nt++] = E1432_TACH_CHAN(2);
	tach_used[1] = 1;
    }
    if (tach3)
    {
	tchan_list[nt++] = E1432_TACH_CHAN(3);
	tach_used[2] = 1;
    }
    if (tach4)
    {
	tchan_list[nt++] = E1432_TACH_CHAN(4);
	tach_used[3] = 1;
    }

    *group = e1432_create_channel_group(*hw, nc + nt, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nmod, int nchan, SHORTSIZ16 *chan_list,
      double interval, int master_tach, int pre_arm)
{
    int tchan;
    double high, low, pre;

    CHECK(e1432_set_append_status(hw, group, E1432_APPEND_STATUS_ON));
    CHECK(e1432_set_arm_mode(hw, group, E1432_ARM_RPM_RUNUP));
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    CHECK(e1432_set_span(hw, group, SPAN));

    /* The standard RPM test setup has tach0 and tach3 match, and
       tach1 and tach2 are half that frequency. */
    high = RPM_HIGH;
    low = RPM_LOW;
    pre = RPM_PRE_ARM;
    if (master_tach == 1 || master_tach == 2)
    {
	high /= 2;
	interval /= 2;
	low /= 2;
	pre /= 2;
    }

    /* Minimal rpm arm setup */
    tchan = E1432_TACH_CHAN(master_tach + 1);
    CHECK(e1432_set_arm_channel(hw, tchan, E1432_CHANNEL_ON));
    CHECK(e1432_set_rpm_high(hw, tchan, high));
    CHECK(e1432_set_rpm_interval(hw, tchan, interval));
    CHECK(e1432_set_rpm_low(hw, tchan, low));
    if (pre_arm)
    {
	CHECK(e1432_set_pre_arm_mode(hw, tchan, E1432_ARM_RPM_RUNUP));
	CHECK(e1432_set_pre_arm_rpm(hw, tchan, pre));
    }

    if (nmod > 1)
	CHECK(e1432_set_trigger_master(hw, tchan, E1432_TRIGGER_MASTER_ON));

    /* Try to handle either TTL levels or +-1V input */
    CHECK(e1432_set_trigger_level(hw, group,
				  E1432_TRIGGER_LEVEL_LOWER, 0.4));
    CHECK(e1432_set_trigger_level(hw, group,
				  E1432_TRIGGER_LEVEL_UPPER, 0.8));

    return 0;
}

static int
check_trailer(struct e1432_trailer *trailer, FLOATSIZ32 clock_freq,
	      double span, int chan, int type)
{
    double  tmp;
    int     df2, df5;

    if (trailer->zoom_corr != 0)
    {
	/* Zoom correction is not currently implemented */
	(void) fprintf(stderr,
		       "%s: trailer zoom corr non-zero: %g (0x%lx)\n",
		       progname, trailer->zoom_corr,
		       *(long *) &trailer->zoom_corr);
	return -1;
    }
    if (trailer->gap < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer gap negative: 0x%lx\n",
		       progname, trailer->gap);
	return -1;
    }
    if (trailer->rpm1 < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm1 negative: %g (0x%lx)\n",
		       progname, trailer->rpm1,
		       *(long *) &trailer->rpm1);
	return -1;
    }
    if (trailer->rpm2 < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm2 negative: %g (0x%lx)\n",
		       progname, trailer->rpm2,
		       *(long *) &trailer->rpm2);
	return -1;
    }
    if (trailer->peak != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer peak non-zero: %g (0x%lx)\n",
		       progname, trailer->peak,
		       *(long *) &trailer->peak);
	return -1;
    }
    if (trailer->rms != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rms non-zero: %g (0x%lx)\n",
		       progname, trailer->rms,
		       *(long *) &trailer->rms);
	return -1;
    }

    /* Compute df2 and df5 from clock_freq and span */
    tmp = span * 2.56;
    df2 = 0;
    df5 = 0;
    while (tmp < clock_freq * 0.9999)
    {
	df2++;
	tmp *= 2;
    }
    if (tmp > clock_freq * 1.0001)
    {
	tmp /= 8;
	tmp *= 5;
	df2 -= 3;
	df5++;
	if (tmp > clock_freq * 1.0001 || tmp < clock_freq * 0.9999)
	{
	    (void) fprintf(stderr,
			   "%s: invalid span/clock_freq combination: %g/%g\n",
			   progname, span, clock_freq);
	    return -1;
	}
    }

    if (df2 != ((trailer->info & E1432_TRAILER_INFO_DEC_2_MASK)
		>> E1432_TRAILER_INFO_DEC_2_SHIFT))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df2 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df2);
	return -1;
    }
    if (df5 != ((trailer->info & E1432_TRAILER_INFO_DEC_5) != 0))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df5 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df5);
	return -1;
    }

    if (((trailer->info & E1432_TRAILER_INFO_CHAN_MASK) >>
	 E1432_TRAILER_INFO_CHAN_SHIFT) != chan - 1)
    {
	(void) fprintf(stderr,
		       "%s: trailer info chan mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, chan - 1);
	return -1;
    }
    if (((trailer->info & E1432_TRAILER_INFO_TYPE_MASK) >>
	 E1432_TRAILER_INFO_TYPE_SHIFT) != type)
    {
	(void) fprintf(stderr,
		       "%s: trailer info type mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, type);
	return -1;
    }

    return 0;
}

static int
check_tach_and_trigger(E1432ID hw, int group, int nmod,
		       int master_tach, int verbose)
{
    SHORTSIZ16 irq_status;
    LONGSIZ32 count;

    if (nmod > 1)
    {
	CHECK(e1432_read_register(hw, E1432_TACH_CHAN(master_tach + 1),
				  E1432_IRQ_STATUS2_REG, &irq_status));

	/* Send trigger to slave modules */
	if ((irq_status & E1432_IRQ_TRIGGER) != 0)
	{
	    if (verbose > 3)
		(void) printf("Calling e1432_send_trigger\n");
	    CHECK(e1432_send_trigger(hw, group));
	}

	/* Send any available master tachs to all modules */
	CHECK(e1432_send_tachs(hw, group,
			       E1432_TACH_CHAN(master_tach + 1),
			       NULL, 0, &count));
	if (count > 0 && verbose > 3)
	    (void) printf("e1432_send_tachs sent %d\n", count);
    }
}

/* Return values:
   0:  Block available
   -2:  Special error that might be OK
   all other negative values:  Normal error, abort program
*/
static int
wait_block_avail(E1432ID hw, int group, int nmod, int master_tach,
		 int scan, int verbose, long blocksize, double span)
{
    clock_t start, timeout;
    int     status;

    timeout = (2 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (verbose > 2)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    while ((status = e1432_block_available(hw, group)) == 0)
    {
	CHECK(check_tach_and_trigger(hw, group, nmod, master_tach,
				     verbose));

	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    /* We can get this error if the rpm runup is just at the
	       peak when we do the init measure.  I guess that's just
	       how it works, though it seems a little bogus to me.
	       Anyway, we don't want to clutter the output with
	       something that may not really be an error.  Return -2
	       so the caller can identify that this is what
	       happenned. */
#if 0
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
#endif
	    return -2;
	}
    }
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_block_available: returned %d\n",
		       progname, status);
	return status;
    }
    if (verbose > 0)
	(void) printf("Scan %d block available found\n", scan);

#if 0
    /* These should NOT be necessary, but they currently are. */
    CHECK(check_tach_and_trigger(hw, group, nmod, master_tach,
				 verbose));
    CHECK(check_tach_and_trigger(hw, group, nmod, master_tach,
				 verbose));
#endif

    return 0;
}

/* This program doesn't keep track of which input channels or which
   tach channels are in which module, so we don't know which tach
   channel's rpm to compare the trailer against.  Rather than deal
   with that, we'll just test the data RPM farther below, which is all
   MeasEngine uses anyway.

   However, we know that the first four input channels come from the
   first input module, so we know to use the first tach channels to
   compare.  But wait, if neither of the first two tach channels are
   active, then the trailer ends up with RPM values from the second
   pair of tach channels. */
static int
check_trailer_rpm(int chan, int *tach_used, double *rpm,
		  struct e1432_trailer *trailer)
{
    double rpmtmp;

    if (chan >= 4)
	return 0;

    if (tach_used[0] || tach_used[1])
	rpmtmp = rpm[0];
    else
	rpmtmp = rpm[2];
    if (trailer->rpm1 != rpmtmp)
    {
	(void) fprintf(stderr,
		       "%s: Trailer RPM1 %g != Data RPM %g\n",
		       progname, trailer->rpm1, rpmtmp);
	return -1;
    }
    if (tach_used[0] || tach_used[1])
	rpmtmp = rpm[1];
    else
	rpmtmp = rpm[3];
    if (trailer->rpm2 != rpmtmp)
    {
	(void) fprintf(stderr,
		       "%s: Trailer RPM2 %g != Data RPM %g\n",
		       progname, trailer->rpm2, rpmtmp);
	return -1;
    }

    return 0;
}

/* Round to nearest interval step above low */
static double
round_to_step(double low, double interval, double rpm)
{
    return low + interval * floor((rpm - low) / interval + 0.5);
}

static int
check_rpm(int tach_used, double rpm, double prev_rpm, double low,
	  double interval, int allow_any_rpm, int exact,
	  int big_rpm_errors, int verbose)
{
    double  expected, errmax;

#if 0
    (void) printf("tach_used %d rpm %g prev_rpm %g low %g "
		  "interval %g allowany %d exact %d\n",
		  tach_used, rpm, prev_rpm, low, interval,
		  allow_any_rpm, exact);
#endif

    if (tach_used)
    {
	if (allow_any_rpm)
	{
	    prev_rpm = rpm - interval;
	    prev_rpm = round_to_step(low, interval, prev_rpm);
	    if (verbose > 2)
		(void) printf("Switching prev_rpm to %g\n", prev_rpm);
	}

	expected = prev_rpm + interval;
	expected = round_to_step(low, interval, expected);

	errmax = exact ? expected * 0.000001 : expected * 0.001;
	if (big_rpm_errors)
	    errmax *= 5;
    }
    else
    {
	expected = 0;
	errmax = 0;
    }

    if (rpm > expected + errmax || rpm < expected - errmax)
    {
	(void) fprintf(stderr, "%s: RPM %g, expected %g, errmax %g\n",
		       progname, rpm, expected, errmax);
	return -1;
    }

    return 0;
}

static int
compare(double d1, double d2, int big_rpm_errors)
{
    double diff, errmax;

    diff = fabs(d1 - d2);

    if (fabs(d1) > fabs(d2))
	errmax = fabs(d1);
    else
	errmax = fabs(d2);

    if (big_rpm_errors)
	errmax *= 0.001;
    else
	errmax *= 0.000001;

    if (diff > errmax)
    {
	(void) fprintf(stderr,
		       "%s: difference too big: %g %g maxerr %g\n",
		       progname, d1, d2, errmax);
	return -1;
    }

    return 0;
}

static int
check_all_rpm(int *tach_used, double *prev_rpm, int big_rpm_errors)
{
    /* The standard RPM test setup has tach0 and tach3 match, and
       tach1 and tach2 are half that frequency. */
    if (tach_used[0])
    {
	if (tach_used[1])
	    CHECK(compare(prev_rpm[0], 2 * prev_rpm[1], big_rpm_errors));
	if (tach_used[2])
	    CHECK(compare(prev_rpm[0], 2 * prev_rpm[2], big_rpm_errors));
	if (tach_used[3])
	    CHECK(compare(prev_rpm[0], prev_rpm[3], big_rpm_errors));
	return 0;
    }

    if (tach_used[1])
    {
	if (tach_used[2])
	    CHECK(compare(prev_rpm[1], prev_rpm[2], big_rpm_errors));
	if (tach_used[3])
	    CHECK(compare(2 * prev_rpm[1], prev_rpm[3], big_rpm_errors));
	return 0;
    }

    if (tach_used[2] && tach_used[3])
	CHECK(compare(2 * prev_rpm[2], prev_rpm[3], big_rpm_errors));
    return 0;
}

static int
run(E1432ID hw, int group, int nmod, int nchan, SHORTSIZ16 *chan_list,
    int master_tach, int *tach_used, double interval, int pre_arm,
    long nscan, int big_rpm_errors, int verbose)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    FLOATSIZ32 frpm;
    LONGSIZ32 count;
    double  high[NTACH_MAX], ival_array[NTACH_MAX], low[NTACH_MAX];
    double  rpm[NTACH_MAX], prev_rpm[NTACH_MAX];
    struct e1432_trailer trailer;
    int     scan, chan, tach, allow_any_rpm, long_wait, status, first;

    /* Zero the RPMs, so that we can compare inactive tach channel RPM
       values with trailer RPM values. */
    for (tach = 0; tach < NTACH_MAX; tach++)
	rpm[tach] = 0;

    allow_any_rpm = !pre_arm;
    long_wait = pre_arm;
    first = 1;

    /* The standard RPM test setup has tach0 and tach3 match, and
       tach1 and tach2 are half that frequency. */
    high[0] = RPM_HIGH;
    high[1] = high[0] / 2;
    high[2] = high[1];
    high[3] = high[0];
    ival_array[0] = interval;
    ival_array[1] = ival_array[0] / 2;
    ival_array[2] = ival_array[1];
    ival_array[3] = ival_array[0];
    low[0] = RPM_LOW;
    low[1] = low[0] / 2;
    low[2] = low[1];
    low[3] = low[0];
    prev_rpm[0] = RPM_LOW - interval;
    prev_rpm[1] = prev_rpm[0] / 2;
    prev_rpm[2] = prev_rpm[1];
    prev_rpm[3] = prev_rpm[0];

    status = e1432_init_measure(hw, group);
    if (status < 0)
    {
	if (!pre_arm && status == ERR1432_STATUS_READ_TIMEOUT)
	{
	    /* It might be legitimate to get a timeout error right
	       after starting the measurement, if the RPM runup is
	       just reaching the top.  It seems a little bogus, but
	       I'll allow it.  In this case, just re-start the
	       measurement.  We assume that this one can't error
	       because the runup is no longer at the top anymore. */
	    if (verbose > 2)
		(void) printf("Restarting - first init_meas timed out\n");
	    first = 0;
	    CHECK(e1432_init_measure(hw, group));
	}
	else
	{
	    fprintf(stderr, "%s: e1432_init_measure: returned %d\n",
		    progname, status);
	    return -1;
	}
    }

    for (scan = 0; scan < nscan; scan++)
    {
	/* Wait for block available.  If first scan and we are
	   pre-arming, allow a much longer wait because the signal
	   sweep may have to get back to the beginning.  Internally,
	   this takes care of the e1432_send_trigger stuff needed for
	   multi-module rpm arming. */
	status = wait_block_avail(hw, group, nmod, master_tach,
				  scan, verbose, BLOCKSIZE,
				  long_wait ? SPAN / 200 : SPAN);
	if (status < 0)
	{
	    if (first && !pre_arm && status == -2)
	    {
		/* It might be legitimate to get a timeout error right
		   after starting the measurement, if the RPM runup is
		   just reaching the top.  It seems a little bogus,
		   but I'll allow it.  In this case, just re-start the
		   measurement.  We assume that this one can't error
		   because the runup is no longer at the top
		   anymore. */
		if (verbose > 2)
		    (void) printf("Restarting - first block timed out\n");
		CHECK(e1432_init_measure(hw, group));
		CHECK(wait_block_avail(hw, group, nmod, master_tach,
				       scan, verbose, BLOCKSIZE,
				       long_wait ? SPAN / 200 : SPAN));
	    }
	    else
	    {
		fprintf(stderr, "%s: wait_block_avail: returned %d%s\n",
			progname, status, status == -2 ? " (timeout)" : "");
		return -1;
	    }
	}

	long_wait = 0;
	first = 0;

	/* Read the DataRPM values, which must be read after block
	   available but before reading the data. */
	for (tach = 0; tach < NTACH_MAX; tach++)
	    if (tach_used[tach])
	    {
		CHECK(e1432_get_data_rpm(hw, E1432_TACH_CHAN(tach + 1),
					 &frpm));
		rpm[tach] = frpm;
		if (verbose > 1)
		    (void) printf("Read tach chan %d data RPM %g\n",
				  tach, rpm[tach]);
	    }

	/* Read the data */
	for (chan = 0; chan < nchan; chan++)
	{
	    if (verbose > 2)
		(void) printf("Reading chan %d\n", chan);

	    CHECK(e1432_read_float64_data(hw, chan_list[chan],
					  E1432_TIME_DATA,
					  buffer, BLOCKSIZE,
					  &trailer,
					  &count));
	    if (count != BLOCKSIZE)
	    {
		(void) fprintf(stderr,
			       "%s: e1432_read_float64_data: "
			       "actual count was %ld\n",
			       progname, count);
		return -1;
	    }

	    CHECK(check_trailer(&trailer, 51200, SPAN,
				chan_list[chan], 0));

	    if (verbose > 2)
	    {
		(void) printf("Chan %d trailer RPM1 %g\n",
			      chan, trailer.rpm1);
		(void) printf("Chan %d trailer RPM2 %g\n",
			      chan, trailer.rpm2);
	    }

	    CHECK(check_trailer_rpm(chan, tach_used, rpm, &trailer));
	}

	/* Check the DataRPM values */
	for (tach = 0; tach < NTACH_MAX; tach++)
	{
	    CHECK(check_rpm(tach_used[tach], rpm[tach],
			    prev_rpm[tach], low[tach],
			    ival_array[tach], allow_any_rpm,
			    tach == master_tach,
			    big_rpm_errors, verbose));

	    prev_rpm[tach] =
		round_to_step(low[tach], ival_array[tach], rpm[tach]);
	}
	/* Use rounded prev_rpm values for this */
	CHECK(check_all_rpm(tach_used, prev_rpm, big_rpm_errors));

	/* Check for end of sweep.  If found, quit measurement now.
	   Detect this by checking the first used tach channel. */
	if (prev_rpm[master_tach] >
	    high[master_tach] - ival_array[master_tach] * 0.95)
	    break;

	allow_any_rpm = 0;
    }

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
	  "Usage: %s [-1234IpuvV] [-i interval] [-L laddr] [-n nchan]\n"
		   "\t[-N nmod] [-s scans]\n"
		   "\t-1: Activate tach channel 1\n"
		   "\t-2: Activate tach channel 2\n"
		   "\t-3: Activate tach channel 3\n"
		   "\t-4: Activate tach channel 4\n"
		   "\t-i: Set RPM interval, default 100\n"
		   "\t-I: Ignore small RPM errors\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-p: Use RPM pre-arm\n"
		   "\t-s: Do <scans> scans, default 100\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    double  interval;
    char   *p;
    long    nscan;
    int     tach_used[NTACH_MAX];
    int     c, i, nmod, pre_arm, verbose, big_rpm_errors;
    int     tach1, tach2, tach3, tach4, master_tach;
    int     group, nchan;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    master_tach = -1;
    tach1 = 0;
    tach2 = 0;
    tach3 = 0;
    tach4 = 0;
    interval = 100;
    big_rpm_errors = 0;
    laddr[0] = 8;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    pre_arm = 0;
    nscan = 100;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "1234i:IL:n:N:ps:uvV")) != -1)
	switch (c)
	{
	case '1':
	    tach1 = 1;
	    if (master_tach < 0)
		master_tach = 0;
	    break;
	case '2':
	    tach2 = 1;
	    if (master_tach < 0)
		master_tach = 1;
	    break;
	case '3':
	    tach3 = 1;
	    if (master_tach < 0)
		master_tach = 2;
	    break;
	case '4':
	    tach4 = 1;
	    if (master_tach < 0)
		master_tach = 3;
	    break;
	case 'i':
	    if (sscanf(optarg, "%lg", &interval) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid RPM interval: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'I':
	    big_rpm_errors = 1;
	    break;
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'p':
	    pre_arm = 1;
	    break;
	case 's':
	    nscan = strtol(optarg, &p, 0);
	    if (optarg == p || nscan < 0)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of scans: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    if (!tach1 && !tach2 && !tach3 && !tach4)
    {
	(void) fprintf(stderr, "%s: activate at least one tach channel\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    if (init(nmod, laddr, &hw, &group, &nchan, chan_list,
	     tach1, tach2, tach3, tach4, tach_used) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nmod, nchan, chan_list,
	      interval, master_tach, pre_arm) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nmod, nchan, chan_list, master_tach, tach_used,
	    interval, pre_arm, nscan, big_rpm_errors, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
